/*
 * Copyright (C) 2012-2017 Japan Smartphone Security Association
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jssec.android.cryptasymmetrickey;

import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;

public final class RsaCryptoAsymmetricKey {

	// *** POINT 1 *** Explicitly specify the encryption mode and the padding.
	// *** POINT 2 *** Use strong encryption methods (specifically, technologies that meet the relevant criteria), including algorithms, block cipher modes, and padding modes..
	// Parameters passed to getInstance method of the Cipher class: Encryption algorithm, block encryption mode, padding rule
	// In this sample, we choose the following parameter values: encryption algorithm=RSA, block encryption mode=NONE, padding rule=OAEPPADDING.
	private static final String TRANSFORMATION = "RSA/NONE/OAEPPADDING";

	// encryption algorithm
	private static final String KEY_ALGORITHM = "RSA";

	// *** POINT 3 *** Use a key of length sufficient to guarantee the strength of encryption.
	// Check the length of the key
	private static final int MIN_KEY_LENGTH = 2000;

	RsaCryptoAsymmetricKey() {
	}

	public final byte[] encrypt(final byte[] plain, final byte[] keyData) {
		byte[] encrypted = null;

		try {
			// *** POINT 1 *** Explicitly specify the encryption mode and the padding.
			// *** POINT 2 *** Use strong encryption methods (specifically, technologies that meet the relevant criteria), including algorithms, block cipher modes, and padding modes..
			Cipher cipher = Cipher.getInstance(TRANSFORMATION);

			PublicKey publicKey = generatePubKey(keyData);
			if (publicKey != null) {
				cipher.init(Cipher.ENCRYPT_MODE, publicKey);
				encrypted = cipher.doFinal(plain);
			}
		} catch (NoSuchAlgorithmException e) {
		} catch (NoSuchPaddingException e) {
		} catch (InvalidKeyException e) {
		} catch (IllegalBlockSizeException e) {
		} catch (BadPaddingException e) {
		} finally {
		}

		return encrypted;
	}

	public final byte[] decrypt(final byte[] encrypted, final byte[] keyData) {
		// In general, decryption procedures should be implemented on the server side;
		// however, in this sample code we have implemented decryption processing within the application to ensure confirmation of proper execution.
		// When using this sample code in real-world applications, be careful not to retain any private keys within the application. 
		
		byte[] plain = null;

		try {
			// *** POINT 1 *** Explicitly specify the encryption mode and the padding.
			// *** POINT 2 *** Use strong encryption methods (specifically, technologies that meet the relevant criteria), including algorithms, block cipher modes, and padding modes..
			Cipher cipher = Cipher.getInstance(TRANSFORMATION);

			PrivateKey privateKey = generatePriKey(keyData);
			cipher.init(Cipher.DECRYPT_MODE, privateKey);

			plain = cipher.doFinal(encrypted);
		} catch (NoSuchAlgorithmException e) {
		} catch (NoSuchPaddingException e) {
		} catch (InvalidKeyException e) {
		} catch (IllegalBlockSizeException e) {
		} catch (BadPaddingException e) {
		} finally {
		}

		return plain;
	}

	private static final PublicKey generatePubKey(final byte[] keyData) {
		PublicKey publicKey = null;
		KeyFactory keyFactory = null;

		try {
			keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
			publicKey = keyFactory.generatePublic(new X509EncodedKeySpec(keyData));
		} catch (IllegalArgumentException e) {
		} catch (NoSuchAlgorithmException e) {
		} catch (InvalidKeySpecException e) {
		} finally {
		}

		// *** POINT 3 *** Use a key of length sufficient to guarantee the strength of encryption.
		// Check the length of the key
		if (publicKey instanceof RSAPublicKey) {
			int len = ((RSAPublicKey) publicKey).getModulus().bitLength();
			if (len < MIN_KEY_LENGTH) {
				publicKey = null;
			}
		}

		return publicKey;
	}

	private static final PrivateKey generatePriKey(final byte[] keyData) {
		PrivateKey privateKey = null;
		KeyFactory keyFactory = null;

		try {
			keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
			privateKey = keyFactory.generatePrivate(new PKCS8EncodedKeySpec(keyData));
		} catch (IllegalArgumentException e) {
		} catch (NoSuchAlgorithmException e) {
		} catch (InvalidKeySpecException e) {
		} finally {
		}

		return privateKey;
	}
}
